package cz.drg.clasificator.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;


public class HeaderList {

    private static final String NO_WRITABLE_DATA = "Writer has no data to write to target location.";
    private static final String NULL_HEADERS_OR_DATA = "Reader returned null for header or valid elements list data! Headers: %s Columns: %s";
    private static final String NULL_READER = "ValidityList reader cannot be null!";
    private static final String UNEQUAL_LENGTH_HEADERS_COLUMNS_DATA = "Unequal number of headers and columns from reader! Headers: %n Columns: %n";
    
    private static final String HEADER_NOT_FOUND = "Header '%s' does not exist in this validity list! Available headers: [%s]";
    
    protected Map<String, Integer> headerToColumn;
    protected List<List<String>> columnsWithoutHeader;
    protected int numberOfLines = 0;
    
    private HeaderListReader reader;

    public HeaderList(HeaderListReader reader) {
        if(reader == null){
            throw new IllegalArgumentException(NULL_READER);
        }
            
        this.reader = reader;
        readData();
    }
    
    private void readData(){
        
        headerToColumn = reader.getHeadersWithIndexes();
        columnsWithoutHeader = reader.getColumnLists();
        numberOfLines = reader.getNumberOfLines();
        checkReadData();
        
    }

    protected void checkReadData() {
        //check if reader returned any data
        if(headerToColumn == null || columnsWithoutHeader == null){
            throw new IllegalArgumentException(String.format(NULL_HEADERS_OR_DATA, headerToColumn, columnsWithoutHeader));
        }

        //check if there is same ammount of headers and columns
        if(numberOfLines != 0 && headerToColumn.size() != columnsWithoutHeader.size()){
            throw new IllegalArgumentException(String.format(UNEQUAL_LENGTH_HEADERS_COLUMNS_DATA, headerToColumn.size(), columnsWithoutHeader.size()));
        }
    }
    
    public boolean hasHeader(String header){
        return headerToColumn.containsKey(header);
    }
    
    public List<String> getColumnList(String header){
        if(!hasHeader(header)){
            throw new IllegalArgumentException(String.format(HEADER_NOT_FOUND, header, headerToColumn.keySet()));
        }
        return columnsWithoutHeader.get(headerToColumn.get(header));
    }
    
    public Integer getHeaderIndex(String header){
        if(!hasHeader(header)){
            throw new IllegalArgumentException(String.format(HEADER_NOT_FOUND, header, headerToColumn.keySet()));
        }
        
        return headerToColumn.get(header);
    }
    
    public String getValue(int index, String header){
        return getColumnList(header).get(index);
    }

    public Map<String, String> getRow(int lineIndex){
        Map<String,String> result = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        
        for (String header : headerToColumn.keySet()) {
            
            result.put(header, getValue(lineIndex, header));
            
        }
        
        return result;
    }
    
    public List<String> getRowAsList(int lineIndex){
        List<String> result = new ArrayList<>();
        
        for (int i = 0; i < columnsWithoutHeader.size(); i++) {
            String rowValue = columnsWithoutHeader.get(i).get(lineIndex);
            result.add(rowValue);
        }
            
        return result;
    }

    public List<Map<String, String>> getRows(){
        
        return getRows(0, numberOfLines);
        
    }
    public List<Map<String, String>> getRows(int lineStartIndex, int linesAmmount){
        List<Map<String, String>> result = new ArrayList<>();
        
        for (int i = 0; i < linesAmmount; i++) {
            
            result.add(getRow(lineStartIndex + i));
            
        }
        
        return result;
    }
    
    public int numOfLines() {
        return numberOfLines;
    }

    public Map<String, Integer> getHeaderToColumn() {
        return headerToColumn;
    }

    public List<List<String>> getColumnsWithoutHeader() {
        return columnsWithoutHeader;
    }

    public List<List<String>> getValues(){
        
        List<List<String>> columsWithHeader = new ArrayList<>();
        columsWithHeader.add(getSortedHeaderList());
        
        for (int i = 0; i < numOfLines(); i++) {
            
            columsWithHeader.add(getRowAsList(i));
            
        }
        
        return columsWithHeader;
        
    }
    
    private List<String> getSortedHeaderList(){
        
        
        
        List<Entry<String, Integer>> headers = new ArrayList<>(headerToColumn.entrySet());
        
        Collections.sort(headers, new Comparator<Entry<String, Integer>>() {

            @Override
            public int compare(Entry<String, Integer> o1, Entry<String, Integer> o2) {
                return o1.getValue().compareTo(o2.getValue());
            }
        });
        
        List<String> headerList = new ArrayList<>();
        
        for (Entry<String, Integer> header : headers) {
            headerList.add(header.getKey());
        }
        
        return headerList;
    }
}
